import { Suspense } from "react"
import { notFound } from "next/navigation"
import { Header } from "@/components/header"
import { TVSeasonsOverview } from "@/components/tv-seasons-overview"
import { Skeleton } from "@/components/ui/skeleton"

async function getTVShowWithSeasons(id: string) {
  try {
    const res = await fetch(`https://api.themoviedb.org/3/tv/${id}?api_key=${process.env.TMDB_API_KEY}`, {
      next: { revalidate: 3600 },
    })

    if (!res.ok) {
      console.error(`TMDB TV show fetch failed (${res.status}) for id ${id}`)
      return null
    }

    return res.json()
  } catch (err) {
    console.error("TMDB TV show fetch error:", err)
    return null
  }
}

export default async function TVSeasonsPage({
  params,
}: {
  params: { id: string }
}) {
  const tvShow = await getTVShowWithSeasons(params.id)

  if (!tvShow) {
    notFound()
  }

  return (
    <div className="min-h-screen bg-background">
      <Header />

      <div className="container mx-auto px-4 py-8">
        <div className="mb-8">
          <h1 className="text-3xl font-bold mb-2">{tvShow.name} - Seasons</h1>
          <p className="text-muted-foreground">
            {tvShow.number_of_seasons} Season{tvShow.number_of_seasons !== 1 ? "s" : ""} • {tvShow.number_of_episodes}{" "}
            Episodes
          </p>
        </div>

        <Suspense fallback={<TVSeasonsOverviewSkeleton />}>
          <TVSeasonsOverview tvShow={tvShow} />
        </Suspense>
      </div>
    </div>
  )
}

function TVSeasonsOverviewSkeleton() {
  return (
    <div className="space-y-6">
      {Array.from({ length: 3 }).map((_, i) => (
        <div key={i} className="flex space-x-4 p-6 border rounded-lg">
          <Skeleton className="w-32 h-48 rounded-lg" />
          <div className="flex-1 space-y-3">
            <Skeleton className="h-6 w-48" />
            <Skeleton className="h-4 w-32" />
            <Skeleton className="h-4 w-full" />
            <Skeleton className="h-4 w-3/4" />
          </div>
        </div>
      ))}
    </div>
  )
}

export async function generateMetadata({
  params,
}: {
  params: { id: string }
}) {
  const tvShow = await getTVShowWithSeasons(params.id)

  if (!tvShow) {
    return {
      title: "TV Show Not Found",
    }
  }

  return {
    title: `${tvShow.name} - Seasons - MoviesWeb`,
    description: `Browse all seasons and episodes of ${tvShow.name}`,
  }
}
